/**
* \file: mount.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <pthread.h>
#include <stdint.h>
#include <stdbool.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/eventfd.h>
#include <errno.h>
#include <libmount/libmount.h>

#include "control/mount.h"
#include "utils/logger.h"

typedef struct thread_context_t thread_context_t;

struct thread_context_t
{
    pthread_t pthread_id;

    partition_t *partition;

    char *mount_point;

    char *mount_src;

    char *mount_fstype;

    unsigned long mount_flags;

    char *mount_options;

    mount_done_callback_t mount_callback_func;

    umount_done_callback_t umount_callback_func;

    error_code_t return_value;

    bool still_working;

    bool pthread_id_valid;

    bool partition_invalid;

    thread_context_t *next_ctx;
};


static thread_context_t *mount_create_thread_context(partition_t *partition, mount_done_callback_t mount_call_back_func,
        umount_done_callback_t umount_call_back_func, const mount_options_t *options);
static error_code_t mount_thread_context_copy_partition_info(thread_context_t *ctx, partition_t *partition);
static error_code_t mount_thread_context_copy_partition_options(thread_context_t *ctx, const mount_options_t *options);
static void mount_free_thread_context(thread_context_t *ctx);

static error_code_t mount_kick_off_thread(thread_context_t *ctx,void *(*thread_func)(void *params));
static void mount_on_watch_event(watch_t *ev_watch, uint32_t event);
static void mount_thread_signal_finished(thread_context_t *ctx);
static void mount_finalize_finished_threads(void);

static error_code_t mount_prepare_mount(partition_t *partition, const mount_options_t *options);
static error_code_t mount_kickoff_mount(thread_context_t *ctx);
static void* mount_do_mount(void *params);

static error_code_t mount_kickoff_umount(thread_context_t *ctx);
static void* mount_do_umount(void *params);

static void mount_clean_ctx_list(void);


// module singleton attributes
static thread_context_t ctx_list_head=
{
        //list head takes only care for the next element, the rest does not matter
        .next_ctx=NULL,
};

static watch_t mount_watch={-1,0,mount_on_watch_event,NULL};

//---------------------------------------------------- members of the API ----------------------------------------------
error_code_t mount_init(void)
{
	error_code_t result=RESULT_OK;
    ctx_list_head.next_ctx=NULL;

    if (mount_watch.pollfd==-1)
    {
        //start with event flag reset, WORK_BLOCKING, NOT SEMAPHOREMODE, NO CLOEXEC
    	mount_watch.pollfd=eventfd(0,0);
        if (mount_watch.pollfd==-1)
            result=RESULT_NORESOURCE;
    }
    else
        result=RESULT_INVALID;

    if (result==RESULT_OK)
    	result=watch_add_event_source(&mount_watch,EPOLLIN);

    return result;
}

error_code_t mount_start_mount(partition_t *partition, const mount_options_t *options,
        mount_done_callback_t callback_func)
{
    error_code_t result=RESULT_OK;
    thread_context_t *ctx;

    ctx=mount_create_thread_context(partition,callback_func, NULL, options);

    if (ctx==NULL)
        result=RESULT_NORESOURCE;

    if (result==RESULT_OK)
        result=mount_prepare_mount(partition,options);

    if (result==RESULT_OK)
        result=mount_kickoff_mount(ctx);

    return result;
}

error_code_t mount_start_unmount(partition_t *partition, umount_done_callback_t callback_func)
{
    error_code_t result=RESULT_OK;
    thread_context_t *ctx;

    ctx=mount_create_thread_context(partition,NULL, callback_func, NULL);
    if (ctx==NULL)
        result=RESULT_NORESOURCE;

    if (result==RESULT_OK)
        result=mount_kickoff_umount(ctx);

    return result;
}

void mount_mark_partition_invalid(partition_t *partition)
{
    thread_context_t *cur_ctx;
    cur_ctx=ctx_list_head.next_ctx;

    while (cur_ctx!=NULL)
    {
        if (cur_ctx->partition==partition)
        {
            //found this ctx and mark it invalid.
            cur_ctx->partition_invalid=true;
            //stop searching
            cur_ctx=NULL;
        }
        else
            //not found, next one
            cur_ctx=cur_ctx->next_ctx;
    }
}

error_code_t mount_unmount_sync(const char *mount_point)
{
    error_code_t result=RESULT_OK;
    struct libmnt_context *libmnt_ctx;

    libmnt_ctx = mnt_new_context();
    if (libmnt_ctx==NULL)
        result=RESULT_NORESOURCE;

    if (result==RESULT_OK)
    {
        mnt_context_set_target(libmnt_ctx, mount_point);
        mnt_context_enable_lazy(libmnt_ctx,true);
        if (mnt_context_umount(libmnt_ctx))
            result=RESULT_UMOUNT_ERR;
    }

    if (libmnt_ctx!=NULL)
        mnt_free_context(libmnt_ctx);

    if (result==RESULT_OK)
        rmdir(mount_point);

    return result;
}

bool mount_is_someone_working(void)
{
    return ctx_list_head.next_ctx!=NULL;
}

void mount_deinit(void)
{
    mount_clean_ctx_list();

    if (mount_watch.pollfd!=-1)
    {
    	(void)watch_remove_event_source(&mount_watch);
        close(mount_watch.pollfd);
        mount_watch.pollfd=-1;
    }
}
//----------------------------------------------------------------------------------------------------------------------

//------------------------------------------------------- private members ----------------------------------------------
static thread_context_t *mount_create_thread_context(partition_t *partition, mount_done_callback_t mount_callback_func,
        umount_done_callback_t umount_callback_func, const mount_options_t *options)
{
    error_code_t result=RESULT_OK;
    thread_context_t *ctx;
    ctx=malloc(sizeof(thread_context_t));

    if(ctx!=NULL)
    {
        ctx->mount_callback_func=mount_callback_func;
        ctx->umount_callback_func=umount_callback_func;
        ctx->pthread_id_valid=false;
        result=mount_thread_context_copy_partition_info(ctx,partition);
        if (result==RESULT_OK)
            result=mount_thread_context_copy_partition_options(ctx,options);
    }

    if (result!=RESULT_OK && ctx!=NULL)
    {
        mount_free_thread_context(ctx);
        ctx=NULL;
    }

    return ctx;
}

static error_code_t mount_thread_context_copy_partition_info(thread_context_t *ctx, partition_t *partition)
{
    //sorry, this is needed to get rid of lint finding: "Custodial pointer ctx (line xxx) has not been freed or returned"
    thread_context_t *ctx2=ctx;

    ctx->partition=partition;
    ctx->mount_point=strdup(partition_get_mountpoint(partition));
    ctx->mount_src=strdup(partition_get_mountsrc(partition));
    ctx->mount_fstype=strdup(partition_get_mountfs(partition));
    ctx->next_ctx=NULL;
    ctx->still_working=false;
    ctx->partition_invalid=false;

    if (ctx2->mount_point==NULL || ctx2->mount_src==NULL || ctx2->mount_fstype==NULL)
        return RESULT_NORESOURCE;

    return RESULT_OK;
}

static error_code_t mount_thread_context_copy_partition_options(thread_context_t *ctx,
        const mount_options_t *options)
{
    error_code_t result=RESULT_OK;

    ctx->mount_options=NULL;

    if(options!=NULL)
    {
        ctx->mount_flags=options->mount_flags;
        if (options->mount_options!=NULL)
        {
            ctx->mount_options=strdup(options->mount_options);
            if (ctx->mount_options==NULL)
                result=RESULT_NORESOURCE;
        }
        else
            ctx->mount_options=NULL;
    }

    return result;
}

static void mount_free_thread_context(thread_context_t *ctx)
{
    if (ctx!=NULL)
    {
        if (ctx->mount_point!=NULL)
            free(ctx->mount_point);

        if (ctx->mount_src!=NULL)
            free(ctx->mount_src);

        if (ctx->mount_fstype!=NULL)
            free(ctx->mount_fstype);

        if (ctx->mount_options!=NULL)
            free(ctx->mount_options);

        if(ctx->pthread_id_valid==true)
        	pthread_join(ctx->pthread_id,NULL);

        free(ctx);
    }
}

static error_code_t mount_kick_off_thread(thread_context_t *ctx,void *(*thread_func)(void *params))
{
    error_code_t result=RESULT_OK;

    //queue ctx into active threads list
    ctx->next_ctx=ctx_list_head.next_ctx;
    ctx_list_head.next_ctx=ctx;

    //mark as running
    ctx->still_working=true;

    //kick it of
    if (pthread_create(&ctx->pthread_id,NULL, thread_func, ctx)!=0)
        result=RESULT_NORESOURCE;
    else
    	ctx->pthread_id_valid=true;

    return result;
}

static void mount_on_watch_event(watch_t *ev_watch, uint32_t events)
{
    int result;
    uint64_t event_cntr;

    //we know our watch
    (void)ev_watch;
    //we only registered for POLLIN
    (void)events;

    if (mount_watch.pollfd!=-1)
    {
        result=read(mount_watch.pollfd,&event_cntr, sizeof(uint64_t));
        if (result==(int)sizeof(uint64_t) && event_cntr!=0)
            mount_finalize_finished_threads();
    }
}

static void mount_thread_signal_finished(thread_context_t *ctx)
{
    uint64_t event_incr=1;
    ctx->still_working=false;
    if (write(mount_watch.pollfd,&event_incr,sizeof(uint64_t)))
    {
    }
}

static void mount_finalize_finished_threads(void)
{
    thread_context_t *prev_ctx;
    thread_context_t *cur_ctx;
    mount_options_t mount_options;

    prev_ctx=&ctx_list_head;
    cur_ctx=ctx_list_head.next_ctx;

    while (cur_ctx!=NULL)
    {
        if (!cur_ctx->still_working)
        {
            //dequeue cur_ctx
            prev_ctx->next_ctx=cur_ctx->next_ctx;

            // check if our partition got invalid due to a media or device removal.
            // If we are invalid, it is important that we don't touch ctx->partition any longer
            // since this pointer is not valid any more. We don't need to call the callback
            // in this case, no one is waiting for us any longer. Since it is unclear, at which state
            // we got invalid, we try all to clean up. We umount the mount point first and
            // then remove the directory. One of both or both calls might fail.
            if (!cur_ctx->partition_invalid)
            {
                //depending on if we processed a mount or a umount, one of both callback function is set.
                //If the caller isn't interested in any callback at all, none might be set.
                //It's impossible (by API) to set both callbacks
                if (cur_ctx->mount_callback_func)
                {
                    mount_options.mount_flags=cur_ctx->mount_flags;
                    mount_options.mount_options=cur_ctx->mount_options;
                    cur_ctx->mount_callback_func(cur_ctx->partition, cur_ctx->return_value,
                            &mount_options);
                }

                if (cur_ctx->umount_callback_func)
                {
                    cur_ctx->umount_callback_func(cur_ctx->partition, cur_ctx->return_value);
                }
            }
            else
                mount_unmount_sync(cur_ctx->mount_point);

            mount_free_thread_context(cur_ctx);

            //move to the next element
            cur_ctx=prev_ctx->next_ctx;
        }
        else
        {
            prev_ctx=cur_ctx;
            cur_ctx=cur_ctx->next_ctx;
        }
    }
}

static error_code_t mount_prepare_mount(partition_t *partition, const mount_options_t *options)
{
    error_code_t result=RESULT_OK;
    bool remounting_request;
    partition_state_t p_state;

    remounting_request=(options->mount_flags & MS_REMOUNT)!=0;
    p_state=partition_get_state(partition);

    //check if we are in right state if a mount or remount is requested
    if ((remounting_request && p_state!=PARTITION_REMOUNTING) ||
            (!remounting_request && p_state!=PARTITION_MOUNTING))
        result=RESULT_INVALID;

    return result;
}

static error_code_t mount_kickoff_mount(thread_context_t *ctx)
{
    return mount_kick_off_thread(ctx,mount_do_mount);
}

static void* mount_do_mount(void *params)
{
    error_code_t result=RESULT_OK;
    thread_context_t *ctx;
    struct libmnt_context *libmount_ctx;
    int mount_result;

    //don't access ctx->partition in this function since it might get invalid at any time
    ctx=(thread_context_t*)params;

    libmount_ctx = mnt_new_context();
    if (libmount_ctx==NULL)
        result=RESULT_NORESOURCE;

    if (result==RESULT_OK)
    {
    	//only in case we are remount, we are setting fstype and source. Otherwise, mounting fails in case of
    	//hfsplus and ntfs
    	if ((ctx->mount_flags & MS_REMOUNT)==0 && strstr(ctx->mount_options,"remount")==NULL)
    	{
    		mnt_context_set_fstype(libmount_ctx, ctx->mount_fstype);
    		mnt_context_set_source(libmount_ctx, ctx->mount_src);
    	}
        mnt_context_set_target(libmount_ctx, ctx->mount_point);
        mnt_context_set_options(libmount_ctx, ctx->mount_options);
        mnt_context_set_mflags(libmount_ctx, ctx->mount_flags);
        mount_result=mnt_context_mount(libmount_ctx);
        if (mount_result!=0)
        {
        	if (mount_result>0)
        		logger_log_error("Mounting of partition %s failed (mount syscall error code: %d).",ctx->mount_src, mount_result);
        	else
        		logger_log_error("Mounting of partition %s failed (libmount internal error code: %d).",ctx->mount_src, mount_result);
            result=RESULT_MOUNT_ERR;

            logger_log_error("libmount used the following:");
            logger_log_error("\tSource: \"%s\"",mnt_context_get_source(libmount_ctx));
            logger_log_error("\tTarget: \"%s\"",mnt_context_get_target(libmount_ctx));
            logger_log_error("\tFile system: \"%s\"",mnt_context_get_fstype(libmount_ctx));
            logger_log_error("\tOptions: \"%s\"",mnt_fs_get_fs_options(mnt_context_get_fs(libmount_ctx)));
        }
    }

    if (libmount_ctx!=NULL)
        mnt_free_context(libmount_ctx);

    ctx->return_value=result;
    //from here on, don't touch ctx anymore since we are here signaling the main thread that it can go on working.
    //The main thread will finally free the memory allocated for ctx.
    mount_thread_signal_finished(ctx);

    pthread_exit(NULL);

    return NULL;
}

static error_code_t mount_kickoff_umount(thread_context_t *ctx)
{
    return mount_kick_off_thread(ctx,mount_do_umount);
}

static void* mount_do_umount(void *params)
{
    thread_context_t *ctx;

    //don't access ctx->partition in this function since it might get invalid at any time
    ctx=(thread_context_t*)params;
    ctx->return_value=mount_unmount_sync(ctx->mount_point);

    //from here on, don't touch ctx anymore since we are here signaling the main thread that it can go on working.
    //The main thread will finally free the memory allocated for ctx.
    mount_thread_signal_finished(ctx);

    pthread_exit(NULL);

    return NULL;
}

static void mount_clean_ctx_list(void)
{
    thread_context_t *cur_ctx;

    cur_ctx=ctx_list_head.next_ctx;
    while(cur_ctx!=NULL)
    {
        ctx_list_head.next_ctx=cur_ctx->next_ctx;
        if (cur_ctx->still_working)
        {
            pthread_cancel(cur_ctx->pthread_id);
        }
        mount_free_thread_context(cur_ctx);
        cur_ctx=ctx_list_head.next_ctx;
    }
    ctx_list_head.next_ctx=NULL;
}
//----------------------------------------------------------------------------------------------------------------------
